<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package Shopay
 */

/*-----------------------------------------------------------------------------------------------------------------------*/
/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function shopay_body_classes( $classes ) {
    // Adds a class of hfeed to non-singular pages.
    if ( ! is_singular() ) {
        $classes[] = 'hfeed';
    }

    // Adds a class of no-sidebar when there is no sidebar present.
    if ( ! is_active_sidebar( 'sidebar-1' ) ) {
        $classes[] = 'no-sidebar';
    }

    $shopay_site_layout = get_theme_mod( 'shopay_site_layout', 'full-width' );
    $classes[] = 'site--'.esc_attr( $shopay_site_layout );

    if ( is_single() ) {
        $shopay_post_sidebar_layout = get_theme_mod( 'shopay_post_sidebar_layout', 'right-sidebar' );
        $classes[] = esc_attr( $shopay_post_sidebar_layout );
    }

    if ( is_page() ) {
        $shopay_page_sidebar_layout = get_theme_mod( 'shopay_page_sidebar_layout', 'right-sidebar' );
        $classes[] = esc_attr( $shopay_page_sidebar_layout );
    }

    if ( is_archive() || is_front_page() ) {
        $shopay_archive_sidebar_layout = get_theme_mod( 'shopay_archive_sidebar_layout', 'right-sidebar' );
        $classes[] = esc_attr( $shopay_archive_sidebar_layout );
    }

    return $classes;
}
add_filter( 'body_class', 'shopay_body_classes' );

/*-----------------------------------------------------------------------------------------------------------------------*/

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function shopay_pingback_header() {
    if ( is_singular() && pings_open() ){
        printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
    }
}
add_action( 'wp_head', 'shopay_pingback_header' );

/*-----------------------------------------------------------------------------------------------------------------------*/

/**
 * Enqueue admin scripts and styles.
 */
function shopay_admin_scripts(){
    global $shopay_theme_version;

    wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/assets/library/font-awesome/css/all.min.css', '', '5.10.2', 'all' );
    wp_enqueue_style( 'shopay-admin-style', get_template_directory_uri() . '/assets/css/mt-admin-style.css', '', esc_attr( $shopay_theme_version ), 'all' );

    wp_enqueue_script( 'shopay-admin-script', get_template_directory_uri() . '/assets/js/mt-admin-scripts.js', array( 'jquery' ), esc_attr( $shopay_theme_version ), true );
}
add_action( 'admin_enqueue_scripts', 'shopay_admin_scripts' );

/*-----------------------------------------------------------------------------------------------------------------------*/

if ( ! function_exists( 'shopay_fonts_url' ) ):

    /**
     * Register Google fonts for Shopay.
     *
     * @return string Google fonts URL for the theme.
     * @since 1.0.0
     */

    function shopay_fonts_url() {

        $fonts_url = '';
        $font_families = array();

        /**
         * Translators: If there are characters in your language that are not supported
         * by Muli font, translate this to 'off'. Do not translate into your own language.
         */
        if ( 'off' !== _x( 'on', 'Muli font: on or off', 'shopay' ) ) {
            $font_families[] = 'Muli:400,500,600,900';
        }

        /**
         * Translators: If there are characters in your language that are not supported
         * by Rubik font, translate this to 'off'. Do not translate into your own language.
         */
        if ( 'off' !== _x( 'on', 'Rubik font: on or off', 'shopay' ) ) {
            $font_families[] = 'Rubik:500,700';
        }

        if ( $font_families ) {
            $query_args = array(
                'family' => urlencode( implode( '|', $font_families ) ),
                'subset' => urlencode( 'latin,latin-ext' ),
            );

            $fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
        }

        return $fonts_url;
    }

endif;

/*-----------------------------------------------------------------------------------------------------------------------*/

/**
 * Enqueue scripts and styles.
 */
function shopay_scripts() {
    global $shopay_theme_version;

    wp_enqueue_style( 'shopay-fonts', shopay_fonts_url(), array(), null );
    wp_enqueue_style( 'preloader-style', get_template_directory_uri() . '/assets/css/mt-preloader.css', array(), esc_attr( $shopay_theme_version ) );
    //wp_enqueue_style( 'slick-style', get_template_directory_uri() . '/assets/library/slick/css/slick.css', array(), '1.8.0' );
    wp_enqueue_style( 'lightslider-style', get_template_directory_uri() . '/assets/library/lightslider/css/lightslider.min.css', array(), '1.1.3' );
    wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/assets/library/font-awesome/css/all.min.css', '', '5.10.2', 'all' );
    wp_enqueue_style( 'animate', get_template_directory_uri() . '/assets/library/animate/animate.css', '', '3.7.2', 'all' );
    wp_enqueue_style( 'shopay-style', get_stylesheet_uri(), '',esc_attr( $shopay_theme_version ) );

    wp_enqueue_style( 'shopay-responsive-style', get_template_directory_uri(). '/assets/css/responsive.css', array(), esc_attr( $shopay_theme_version) );

    wp_enqueue_script( 'shopay-combine', get_template_directory_uri() . '/assets/js/mt-combine-scripts.js', array( 'jquery' ), esc_attr( $shopay_theme_version ), true );
    wp_enqueue_script( 'shopay-navigation', get_template_directory_uri() . '/assets/js/navigation.js', array(), esc_attr( $shopay_theme_version ), true );
    wp_enqueue_script( 'shopay-skip-link-focus-fix', get_template_directory_uri() . '/assets/js/skip-link-focus-fix.js', array(), esc_attr( $shopay_theme_version ), true );

    $shopay_sticky_menu_option = get_theme_mod( 'shopay_sticky_menu_option', true );
    if ( true === $shopay_sticky_menu_option ) {
        $sticky_value = 'on';
        wp_enqueue_script( 'header-sticky-scripts', get_template_directory_uri() . '/assets/library/sticky/jquery.sticky.min.js', array(), '1.0.4', true );
    } else {
        $sticky_value = 'off';
    }

    $shopay_sticky_sidebar_option = get_theme_mod( 'shopay_sticky_sidebar_option', true );
    if ( true === $shopay_sticky_sidebar_option ) {
        $sidebar_sticky = 'on';
        wp_enqueue_script( 'thia-sticky-sidebar', get_template_directory_uri().'/assets/library/sticky-sidebar/theia-sticky-sidebar.min.js', array(), '1.7.0', true );
    } else {
        $sidebar_sticky = 'off';
    }
    
    $shopay_wow_option = get_theme_mod( 'shopay_wow_option', true );
    if ( true === $shopay_wow_option ) {
        $wow_option = 'on';
        wp_enqueue_script( 'shopay-wow', get_template_directory_uri() . '/assets/library/wow/wow.js', array( 'jquery' ), '1.1.3', true );
    } else {
        $wow_option = 'off';
    }

    wp_enqueue_script( 'shopay-woocommerce', get_template_directory_uri() . '/assets/js/mt-custom-woocommerce.js', array( 'jquery' ), esc_attr( $shopay_theme_version ), true );
    wp_enqueue_script( 'shopay-custom', get_template_directory_uri() . '/assets/js/mt-custom-scripts.js', array( 'jquery' ), esc_attr( $shopay_theme_version ), true );
    
    wp_localize_script( 'shopay-custom', 'shopayObject', array(
        'sidebar_sticky'=> $sidebar_sticky,
        'header_sticky' => $sticky_value,
        'wow_option'    => $wow_option
    ) );
    
    if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
        wp_enqueue_script( 'comment-reply' );
    }
}
add_action( 'wp_enqueue_scripts', 'shopay_scripts' );

/*-----------------------------------------------------------------------------------------------------------------------*/

if ( ! function_exists( 'shopay_social_media' ) ):

    /**
     * Social Icons classes from customizer field.
     */
    function shopay_social_media(){
        $get_shopay_social_icons          = get_theme_mod( 'shopay_social_media', '' );
        $get_decode_shopay_social_icons   = json_decode( $get_shopay_social_icons );
        if ( empty( $get_decode_shopay_social_icons ) ){
            return;
        }
        $social_link_target = get_theme_mod( 'shopay_social_link_target', 'new' );
        if ( 'new' == $social_link_target ) {
            $target = '_blank';
        } else {
            $target = '_self';
        }
    ?>
        <div id="shopay-follow-icons-wrapper">
            <?php 
                foreach ( $get_decode_shopay_social_icons as $key => $value ) {
                    $social_icon_class  = $value->mt_item_icon;
                    $social_icon_url    = $value->mt_item_link;
                    echo '<div class="follow-us-icon"><a href="'.esc_url( $social_icon_url ).'" target="'. esc_attr( $target ) .'"><i class="'.esc_attr( $social_icon_class ).'" aria-hidden="true"></i></a></div>';
                }
            ?>
        </div><!-- .shopay-follow-icons-wrapper -->
<?php
    }

endif;

/*-----------------------------------------------------------------------------------------------------------------------*/

if ( ! function_exists( 'shopay_breadcrumbs' ) ) :

    /**
     * Function for shopay breadcrumbs
     */
    function shopay_breadcrumbs() {
        $shopay_breadcrumbs = get_theme_mod( 'shopay_breadcrumbs', true );
        if ( false == $shopay_breadcrumbs || is_front_page() ) {
            return;
        }
        
        if ( shopay_is_active_woocommerce() ) {
            $bread_home_text = get_theme_mod( 'shopay_breadcrumbs_home_lable', __( 'Home', 'shopay' ) );
            $args = array (
                'wrap_before'   => '<div class="woocommerce-breadcrumbs"> <div class="mt-container"> <div class="woocommerce-breadcrumbs-wrapper">',
                'wrap_after'    => '</div> </div> </div>',
                'home'          => $bread_home_text
            );
            woocommerce_breadcrumb( $args );
        } else {
            if ( ! function_exists( 'breadcrumb_trail' ) ) {
                require_once get_template_directory() . '/inc/class-breadcrumbs.php';
            }

            $breadcrumb_args = array (
                'container'   => 'div',
                'before'      => '<div class="mt-container">',
                'after'       => '</div>',
                'show_browse' => false,
            );
            breadcrumb_trail( $breadcrumb_args );
        }
    }

endif;

add_action( 'shopay_after_header', 'shopay_breadcrumbs', 10 );

/*---------------------------------------------------------------------------------------------------------------*/

if ( ! function_exists( 'shopay_breadcrumbs_labels' ) ) :

    /**
     * Custom breadcrumbs labels
     *
     * @since 1.0.0
     */
    function shopay_breadcrumbs_labels( $defaults ) {

        $defaults['home'] = get_theme_mod( 'shopay_breadcrumbs_home_lable', __( 'Home', 'shopay' ) );

        return $defaults;
    }

endif;

add_filter( 'breadcrumb_trail_labels', 'shopay_breadcrumbs_labels' );
/*-----------------------------------------------------------------------------------------------------------------------*/
if ( !function_exists( 'shopay_category_menu_sec' ) ) :

    /**
     * Function for displaying content of category menu. 
     */
    function shopay_category_menu_sec() {

        $cat_menu_args = array(
            'taxonomy'      => 'product_cat',
            'title_li'      => '',
            'hierarchical'  => true,
            'hide_empty'    => '1',
        );
        echo '<ul class="product-categories">';
            wp_list_categories( apply_filters( 'shopay_slider_cat_list', $cat_menu_args ) );
        echo '</ul>';

    }

endif;

/*-----------------------------------------------------------------------------------------------------------------------*/
if ( ! function_exists( 'shopay_css_strip_whitespace' ) ) :
    
    /**
     * Get minified css and removed space
     *
     * @since 1.0.0
     */

    function shopay_css_strip_whitespace( $css ) {
        $replace = array(
            "#/\*.*?\*/#s" => "",  // Strip C style comments.
            "#\s\s+#"      => " ", // Strip excess whitespace.
        );
        $search = array_keys( $replace );
        $css = preg_replace( $search, $replace, $css );

        $replace = array(
            ": "  => ":",
            "; "  => ";",
            " {"  => "{",
            " }"  => "}",
            ", "  => ",",
            "{ "  => "{",
            ";}"  => "}", // Strip optional semicolons.
            ",\n" => ",", // Don't wrap multiple selectors.
            "\n}" => "}", // Don't wrap closing braces.
            "} "  => "}\n", // Put each rule on it's own line.
        );

        $search = array_keys( $replace );
        $css = str_replace( $search, $replace, $css );

        return trim( $css );
    }

endif;

/*-----------------------------------------------------------------------------------------------------------------------*/

if ( ! function_exists( 'shopay_font_awesome_social_icon_array' ) ) :

    /**
     * Define font awesome social icons
     *
     * @return array();
     * @since 1.0.0
     */
    function shopay_font_awesome_social_icon_array(){
        return array(
            "fab fa-tumblr-square",
            "fab fa-tumblr",
            "fab fa-facebook-square",
            "fab fa-facebook-messenger",
            "fab fa-facebook-f",
            "fab fa-facebook",
            "fab fa-linkedin-in",
            "fab fa-linkedin",
            "fab fa-instagram",
            "fab fa-pinterest-square",
            "fab fa-pinterest-p",
            "fab fa-pinterest",
            "fab fa-whatsapp-square",
            "fab fa-whatsapp",
            "fab fa-twitter-square",
            "fab fa-twitter",
            "fab fa-flickr",
            "fab fa-snapchat-square",
            "fab fa-snapchat-ghost",
            "fab fa-snapchat",
            "fab fa-viber",
        );
    }
endif;

/*----------------------------------------------------------------------------------------------------------------------*/ 

if ( ! function_exists( 'shopay_hover_color' ) ) :

    /**
     * Generate darker color
     * Source: http://stackoverflow.com/questions/3512311/how-to-generate-lighter-darker-color-with-php
     *
     * @since 1.0.0
     */
    function shopay_hover_color( $hex, $steps ) {
        // Steps should be between -255 and 255. Negative = darker, positive = lighter
        $steps = max( -255, min( 255, $steps ) );

        // Normalize into a six character long hex string
        $hex = str_replace( '#', '', $hex );
        if ( strlen( $hex ) == 3) {
            $hex = str_repeat( substr( $hex,0,1 ), 2 ).str_repeat( substr( $hex, 1, 1 ), 2 ).str_repeat( substr( $hex,2,1 ), 2 );
        }

        // Split into three parts: R, G and B
        $color_parts = str_split( $hex, 2 );
        $return = '#';

        foreach ( $color_parts as $color ) {
            $color   = hexdec( $color ); // Convert to decimal
            $color   = max( 0, min( 255, $color + $steps ) ); // Adjust color
            $return .= str_pad( dechex( $color ), 2, '0', STR_PAD_LEFT ); // Make two char hex code
        }

        return $return;
    }

endif;

/**
 * Dynamic style file include
 */
require get_template_directory() . '/inc/dynamic-styles.php';

/**
 * Load Custom Hooks.
 */
require get_template_directory() . '/inc/custom-hooks/header-hooks.php';
require get_template_directory() . '/inc/custom-hooks/custom-hooks.php';
require get_template_directory() . '/inc/custom-hooks/footer-hooks.php';



/**
 * Function for displaying menu item description
 * 
 */
function shopay_nav_description( $item_output, $item, $depth, $args ) {
    if ( ! empty( $item->description ) ) {
        $item_output = str_replace( $args->link_after . '</a>', '<span class="menu-item-description">' . $item->description . '</span>' . $args->link_after . '</a>', $item_output );
    }
    return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'shopay_nav_description', 10, 4 );

/**
 * Check if woocommerce is activated.
 */
function shopay_is_active_woocommerce() {
    if ( class_exists( 'WooCommerce' ) ) {
        return true;
    } else {
        return false;
    }
}

/**
 * Check if wishlist is activated.
 */
function shopay_is_active_wishlist() {
    if ( function_exists( 'YITH_WCWL' ) ) {
        return true;
    } else {
        return false;
    }
}

/**
 * Check if Quick View is activated.
 */
function shopay_is_active_quick_view() {
    if ( class_exists( 'YITH_WCQV_Frontend' ) ) {
        return true;
    } else {
        return false;
    }
}

/**
 * function to display search bar type
 */
function shopay_search_bar_type_choices() {
    if ( shopay_is_active_woocommerce() ) {
        $choices = array(
            'default-search'    => __( 'Default Search', 'shopay' ),
            'product-search'    => __( 'Advance Product Search', 'shopay' )
        );
    } else {
        $choices = array(
            'default-search'    => __( 'Default Search', 'shopay' )
        );
    }

    return $choices;
}

/*---------------------------------------------------------------------------------------------------------------*/
/**
 * One click demo import required functions
 */

// Disable PT branding.
add_filter( 'pt-ocdi/disable_pt_branding', '__return_true' );

if ( ! function_exists( 'shopay_ocdi_after_import' ) ) :

    /**
     * OCDI after import.
     */
    function shopay_ocdi_after_import() {

        // Assign front page
        $front_page_id = null;
        $blog_page_id  = null;

        // Assign navigation menu locations.
        $top_menu   = get_term_by( 'name', 'Top Menu', 'nav_menu' );
        $main_menu  = get_term_by( 'name', 'Main Menu', 'nav_menu' );

        set_theme_mod( 'nav_menu_locations', array(
                'menu-primary' => $main_menu->term_id, // replace 'main-menu' here with the menu location identifier from register_nav_menu() function
                'menu-top' => $top_menu->term_id
            )
        );
    }

endif;

add_action( 'pt-ocdi/after_import', 'shopay_ocdi_after_import' );